import numpy as np

class Classifier:
    
    def __init__(self):
        self.classes = {}
        self.centroids = {}
    
    def addToClass(self, haralick, key):
        cls = self.classes.get(key)
        if cls is None:
            self.classes[key] = np.array(haralick.getFeatureVector())
        else:
            self.classes[key] = np.vstack((cls, haralick.getFeatureVector()))
        #update centroids
        self.centroids[key] = np.mean(self.classes[key], 0)
            
    def classifyByCentroid(self, haralick):
        distances = {}
        v2 = haralick.getFeatureVector()
        #calculate distance to between v2 and each centroid
        for key, centroid in self.centroids.iteritems():
            distances[key] = self.distance(centroid, v2)
        return min(distances, key=lambda k: distances.get(k))
    
    def classifyBySingleNN(self, haralick):
        min_distances = {}
        v2 = haralick.getFeatureVector()
        #calculate distance to between v2 and each point
        for key, cls in self.classes.iteritems():
            cls_distances = []
            for fv in cls:
                cls_distances.append(self.distance(fv, v2))
            min_distances[key] = min(cls_distances)
        return min(min_distances, key=lambda k: min_distances.get(k))
    
    def distance(self, v1, v2):
        return np.linalg.norm(v1 - v2)



class Haralick:
    
    def __init__(self, region, bins, alpha_dist, g_max=255.0):
        self.bins = bins
        self.region = region
        self.featureVector = []
        
        
        #co-occurence matrix for each alpha, distance pair    
        for alpha, dist in alpha_dist:
            c = CoOccMatrix(self.region, alpha, dist, self.bins, g_max)
            
            #features for each matrix
            entr = self.entropy(c)
            contr = self.contrast(c)
            corr = self.correlation(c)
            
            self.featureVector.append(entr)
            self.featureVector.append(contr)
            self.featureVector.append(corr)
            
        self.featureVector = np.array(self.featureVector)
        
        print "Haralick-Features: ", self.featureVector
    
    def getFeatureVector(self):
        return self.featureVector        
    
    def entropy(self, coccMatrix):
        cm = coccMatrix.asArray() 
        m = cm[np.where(cm)]
        e = -np.sum(m * np.log2(m))
        return e

    def contrast(self, coccMatrix):
        cm = coccMatrix.asArray() 
        gs = np.where(cm)
        g1 = gs[0]
        g2 = gs[1]
        
        contr = np.sum(((g1 - g2) ** 2) * cm[gs])
        return contr
    
    def correlation(self, coccMatrix):
        cm = coccMatrix.asArray()          
        g1s, g2s = cm.shape
        
        u1 = 0
        g2_sums = np.arange(g1s, dtype='float')
        for g1 in xrange(g1s):
            s = np.sum(cm[g1, ])
            u1 += g1 * s
            g2_sums[g1] = s
        
        u2 = 0
        g1_sums = np.arange(g2s, dtype='float')
        for g2 in xrange(g2s):
            s = np.sum(cm[:, g2])
            u2 += g2 * s
            g1_sums[g2] = s
    
        sigma1 = 0 
        for g1 in xrange(g1s):  
            sigma1 += ((g1 - u1) ** 2) * g2_sums[g1]    
        sigma1 = np.sqrt(sigma1)
        
        sigma2 = 0 
        for g2 in xrange(g2s):  
            sigma2 += ((g2 - u2) ** 2) * g1_sums[g2]    
        sigma2 = np.sqrt(sigma2)
        
        acc = 0
        for g1 in xrange(g1s):
            for g2 in xrange(g2s):
                acc += (g1 - u1) * (g2 - u2) * cm[g1, g2]
        corr = acc / (sigma1 * sigma2)
        
        return corr
       
    
class CoOccMatrix:
    
    def __init__(self, region, alpha, dist, bins, g_max=255.0):
        self.g_max = g_max
        self.bins = bins
        self.region = region
        a = np.radians(alpha)
        self.offset_x = np.round(np.cos(a) * dist)
        self.offset_y = np.round(np.sin(a) * dist)
        
        h, w = self.region.shape
        self.matrix = np.zeros((self.bins, self.bins))
        
        for x in xrange(w):
            for y in xrange(h):
                g1 = self.getBin(region[y, x])
                g2 = self.getBin(self.getCoForP1((y, x)))
                self.matrix[g1, g2] += 1
        
        self.matrix /= np.sum(self.matrix)
        #calculate features
        
    def get(self, g2, g1):
        return self.matrix[g2, g1]
    
    def asArray(self):
        return self.matrix
    
            
    def getBin(self, value):
        return np.round(value / self.g_max * self.bins - 1) 
    
    def getCoForP1(self, p1):
        y, x = p1
        p2 = (y + self.offset_y, x + self.offset_x)
        p2 = self.normalizeCoord(p2)
        
        return self.region[p2]
    
    def normalizeCoord(self, p):
        #repeat region
        y, x = p
        h, w = self.region.shape
        
        y = (y + h) % h
        x = (x + w) % w
        
        return (y, x)
