package fxml;

import javafx.event.ActionEvent;
import javafx.fxml.FXML;
import javafx.fxml.Initializable;
import javafx.scene.control.RadioButton;
import javafx.scene.control.Slider;
import javafx.scene.control.ToggleGroup;
import javafx.scene.effect.Effect;
import javafx.scene.effect.GaussianBlur;
import javafx.scene.effect.SepiaTone;
import javafx.scene.image.Image;
import javafx.scene.image.ImageView;
import javafx.stage.FileChooser;
import javafx.stage.Stage;

import java.io.File;
import java.net.URL;
import java.util.ResourceBundle;

/**
 * Created by tobias on 18.05.2016.
 */
public class FilterController implements Initializable {

    @FXML private Slider effectStrengthSlider;
    @FXML private ImageView img;
    @FXML private RadioButton blurRadio;
    @FXML private RadioButton sepiaRadio;

    private Stage stage;

    private SepiaTone sepiaTone = new SepiaTone();
    private GaussianBlur gaussianBlur = new GaussianBlur();
    private Effect currentEffect;

    private final int imgWidth = 800, imgHeight = 600;
    private final boolean preserveImageRatio = true, useImageSmoothing = true;

    // the maximum value of the GaussianBlur's radius property is 63
    // https://docs.oracle.com/javafx/2/api/javafx/scene/effect/GaussianBlur.html#radiusProperty()
    private final double blurMax = 63.0;

    // the sepia effect is scaled from 0 to 1
    // https://docs.oracle.com/javafx/2/api/javafx/scene/effect/SepiaTone.html#levelProperty()
    private final double sepiaMax = 1.0;

    @FXML protected void chooseFile(ActionEvent e){
        try{
            FileChooser fileChooser = new FileChooser();
            fileChooser.getExtensionFilters().add(new FileChooser.ExtensionFilter("JPG", "*.jpg")); // only jpg
            fileChooser.getExtensionFilters().add(new FileChooser.ExtensionFilter("PNG", "*.png")); //only png

            // this actually shows the dialog.
            File loadedFile = fileChooser.showOpenDialog(null);

            // we need to convert the file into a URL, as String.
            String imgURL = loadedFile.toURI().toURL().toString();

            // construct the new Image
            // construct the new Image
            Image loadedImg = new Image(imgURL, imgWidth, imgHeight, preserveImageRatio, useImageSmoothing);
            // now update the image view
            img.setImage(loadedImg);

            // immediately apply the effect, if we have one.
            if(currentEffect != null) {
                img.setEffect(currentEffect);
            }
            if(stage != null){
                stage.sizeToScene();
            }

        }catch (Exception exception){
            System.out.println(exception.getMessage());
        }
    }

    @FXML protected void handleRadioButton(ActionEvent e){

        double relativeSliderValue = effectStrengthSlider.getValue() / effectStrengthSlider.getMax();
        if (sepiaRadio.isSelected()) {
            effectStrengthSlider.setMax(sepiaMax);
            currentEffect = sepiaTone;
        }
        else if (blurRadio.isSelected()){
            effectStrengthSlider.setMax(blurMax);
            currentEffect = gaussianBlur;
        }
        // now we shift the slider back to where it was before, only on a different scale.
        // this is usually so quick that the user does not see it.
        effectStrengthSlider.setValue(relativeSliderValue * effectStrengthSlider.getMax());
        // apply the newly resolved effect.
        img.setEffect(currentEffect);
    }


    @Override
    public void initialize(URL location, ResourceBundle resources) {
        this.effectStrengthSlider.setMax(this.sepiaMax);
        this.effectStrengthSlider.setValue(this.effectStrengthSlider.getValue() / 2);

        // we connect the scrollbar's value to the strength of the effect.
        gaussianBlur.radiusProperty().bindBidirectional(effectStrengthSlider.valueProperty());
        sepiaTone.levelProperty().bindBidirectional(effectStrengthSlider.valueProperty());

        ToggleGroup toggleGroup = new ToggleGroup();
        blurRadio.setToggleGroup(toggleGroup);
        sepiaRadio.setToggleGroup(toggleGroup);
    }

    public void setStage(Stage stage){
        this.stage = stage;
    }
}
